
//*****************************************************************************
//
// Author: Stephen Bolanos
// Contact: bstephen94@gmail.com
// Date written: 2015 June 14
//
// The code below measures the duty cycle of a square wave input through pins
// PM1 and PM0 and stores them in 
// 
//*****************************************************************************
#include <stdint.h>
#include <stdbool.h>
#include <math.h>

#include "inc/hw_ints.h"
#include "inc/hw_memmap.h"
#include "inc/hw_types.h"

#include "driverlib/debug.h"
#include "driverlib/gpio.h"
#include "driverlib/interrupt.h"
#include "driverlib/pin_map.h"
#include "driverlib/rom.h"
#include "driverlib/rom_map.h"
#include "driverlib/sysctl.h"
#include "driverlib/timer.h"

#include "timer_configuration.h"

//*****************************************************************************
//
// Variable declarations for use in the program.
//
//*****************************************************************************
uint32_t firstOn, secondFlag, allCaptured;              // Flags
uint32_t firstEdge, secondEdge, thirdEdge;              // Timer value storage when edge event occurs

uint32_t rollOverFlag;                                  // Flag for high time rollover counter
uint32_t highRollOverCount, periodRollOverCount;        // Rollover value storage when timeout event occurs

uint32_t wavePeriod, waveHiTime;                        // Calculated value storage
double waveDutyCycle;

//*****************************************************************************
//
// The error routine that is called if the driver library encounters an error.
//
//*****************************************************************************
#ifdef DEBUG
void
__error__(char *pcFilename, uint32_t ui32Line)
{
}
#endif

//*****************************************************************************
//
// The interrupt handler for timer 2A.
//
//*****************************************************************************
void
Timer2AIntHandler(void)
{
	// Ensures that the first edge captured when reset/turned on will be a rising edge.
    if (firstOn == 0)
    {  
    	// Capture first edge timer value and indicate that the program has captured first on
        firstEdge = MAP_TimerValueGet(TIMER2_BASE, TIMER_A);
        firstOn = 1;
    }
    // Once falling edge has been captured, store the value of the second rising edge which completes the sample
    else if (secondFlag == 1)
    {
        thirdEdge = MAP_TimerValueGet(TIMER2_BASE, TIMER_A);
        // Indicates that all three edges required for calculations have been capture and is ready to calculate
        allCaptured = 1;
    } 

    // For each rising edge, enable the program to count rollOver events for use in calculations.
    rollOverFlag = 1;    

    if (allCaptured == 1)
    {
    	// Reset flags for next sampling
        secondFlag = allCaptured = 0;

        // Duty cycle calculations
        wavePeriod = firstEdge - (thirdEdge - (pow(2, 16) * periodRollOverCount));
        waveHiTime = firstEdge - (secondEdge - (pow(2, 16) * highRollOverCount));
        waveDutyCycle = ((double) waveHiTime/(double) wavePeriod) * 100.00;

        // Store most recent rise edge time as a new first edge to start new sampling.
        firstEdge = thirdEdge;

        // Reset rollOver counts for next sample.
        periodRollOverCount = highRollOverCount = 0;
    }

    // Clear interrupt flag to enable to interrupt to occur once more.
    MAP_TimerIntClear(TIMER2_BASE, TIMER_CAPA_EVENT);
}

//*****************************************************************************
//
// The interrupt handler for timer 2B.
//
//*****************************************************************************
void
Timer2BIntHandler(void)
{
	// Ensures that the first rise edge has been captured and that the falling edge has not yet been captured.
    if (secondFlag != 1 && firstOn == 1)
    {
    	// If fall edge has not yet been captured, store the new timer value and indicate that it has
        secondEdge = MAP_TimerValueGet(TIMER2_BASE, TIMER_B);
        secondFlag = 1;
        // Disable rollover counter for high time sample.
        rollOverFlag = 0;
    }

    // Clear interrupt flag to enable to interrupt to occur once more.
    MAP_TimerIntClear(TIMER2_BASE, TIMER_CAPB_EVENT);
}

//*****************************************************************************
//
// The interrupt handler for timer 3A.
//
//*****************************************************************************
void
Timer3AIntHandler(void)
{
    // Counts rollovers during high time to aid in calculating for high time.
    if (rollOverFlag == 1)
    {
        highRollOverCount ++;
    }

    // Counts all rollovers during the entire period of the square wave.
    periodRollOverCount ++;

    // Clear interrupt flag to enable to interrupt to occur once more.
    MAP_TimerIntClear(TIMER3_BASE, TIMER_TIMA_TIMEOUT);
}

//*****************************************************************************
//
// This example application demonstrates the use of the timers to generate
// periodic interrupts and edge time capture interrupts.
//
//*****************************************************************************
int
main(void)
{   
    two_timer_init();       // Initiates system clock and configures used timers.
    while(1);               // Poll for interrupts
}
